// BorderTron3000 - SAM Coup Edition
//
// Copyright (c)2012 Chris Cowley <chrrris@gmail.com>
//
// Source Licence Terms:
//         1. Please credit me if you re-use this code.
//         2. If you find a way of making money from it, please send me some.
//
// frmMainWnd.pas - Contains the entirety of the user interface, the routine
//                  for creating a Z80 Assembly Listing (BuildASMLineInt()),
//                  and also SaveDSK() which calls the assembler to build the
//                  listing in-memory, and saves it out to disk, wrapped up in
//                  the EDSK file structure.
unit frmMainWnd;

interface

uses
  System.SysUtils, System.Types, System.UITypes, System.Classes, System.Variants,
  FMX.Types, FMX.Controls, FMX.Forms, FMX.Dialogs, FMX.Colors, FMX.ListBox,
  FMX.Objects, FMX.Edit;

type
  BorderArray = array [0..47,0..23] of byte;

  TMainWnd = class(TForm)
    grpEditMode: TGroupBox;
    optColorMode: TRadioButton;
    optSizeMode: TRadioButton;
    grpTopBorder: TGroupBox;
    grpBtmBorder: TGroupBox;
    grpCreate: TGroupBox;
    cmdTopClear: TButton;
    cmdTopLoad: TButton;
    cmdTopSave: TButton;
    pnlTopA: TPanel;
    pnlTopB: TPanel;
    pnlTopD: TPanel;
    pnlTopE: TPanel;
    pnlTopH: TPanel;
    pnlTopL: TPanel;
    pnlBtmA: TPanel;
    pnlBtmL: TPanel;
    pnlBtmH: TPanel;
    pnlBtmE: TPanel;
    pnlBtmD: TPanel;
    pnlBtmB: TPanel;
    lblPalette: TLabel;
    lblBtmPalette: TLabel;
    cmdBtmSave: TButton;
    cmdBtmLoad: TButton;
    cmdBtmClear: TButton;
    pbxTopBorder: TPaintBox;
    pbxBtmBorder: TPaintBox;
    lblLeftMBTop: TLabel;
    cboTopLB: TComboBox;
    lbiTopColA: TListBoxItem;
    lbiTopColB: TListBoxItem;
    lbiTopColD: TListBoxItem;
    lbiTopColE: TListBoxItem;
    lbiTopColH: TListBoxItem;
    lbiTopColL: TListBoxItem;
    lblRightMBTop: TLabel;
    cboTopRB: TComboBox;
    ListBoxItem1: TListBoxItem;
    ListBoxItem2: TListBoxItem;
    ListBoxItem3: TListBoxItem;
    ListBoxItem4: TListBoxItem;
    ListBoxItem5: TListBoxItem;
    ListBoxItem6: TListBoxItem;
    lblRightMBBtm: TLabel;
    cboBtmLB: TComboBox;
    lbiBtmColA: TListBoxItem;
    lbiBtmColB: TListBoxItem;
    lbiBtmColD: TListBoxItem;
    lbiBtmColE: TListBoxItem;
    lbiBtmColH: TListBoxItem;
    lbiBtmColL: TListBoxItem;
    lblLeftMBBtm: TLabel;
    cboBtmRB: TComboBox;
    ListBoxItem7: TListBoxItem;
    ListBoxItem8: TListBoxItem;
    ListBoxItem9: TListBoxItem;
    ListBoxItem10: TListBoxItem;
    ListBoxItem11: TListBoxItem;
    ListBoxItem12: TListBoxItem;
    dlgLoad: TOpenDialog;
    dlgSave: TSaveDialog;
    pnlPickColor: TPanel;
    pnlBlack: TPanel;
    pnlBlue: TPanel;
    pnlRed: TPanel;
    pnlMagenta: TPanel;
    pnlGreen: TPanel;
    pnlCyan: TPanel;
    pnlYellow: TPanel;
    pnlWhite: TPanel;
    pnlBriBlack: TPanel;
    pnlBriBlue: TPanel;
    pnlBriRed: TPanel;
    pnlBriMagenta: TPanel;
    pnlBriGreen: TPanel;
    pnlBriCyan: TPanel;
    pnlBriYellow: TPanel;
    pnlBriWhite: TPanel;
    lblOrg: TLabel;
    txtOrg: TEdit;
    chkExitKeypress: TCheckBox;
    cmdSaveASM: TButton;
    cmdSaveDSK: TButton;
    lblVersionInfo: TLabel;
    lneTopLeftOScan: TLine;
    lblTopLeftOScan: TLabel;
    lblTopLeftTVVisible: TLabel;
    lneTopRightOScan: TLine;
    rctTopPaper: TRectangle;
    lneTopLeftBorder: TLine;
    lneTopRightBorder: TLine;
    lblPaperArea: TLabel;
    procedure lbiTopColAPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure pnlColorPalettePaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiTopColBPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiTopColDPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiTopColEPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiTopColHPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiTopColLPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiBtmColAPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiBtmColBPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiBtmColDPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiBtmColEPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiBtmColHPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure lbiBtmColLPaint(Sender: TObject; Canvas: TCanvas;
      const ARect: TRectF);
    procedure pbxTopBorderPaint(Sender: TObject; Canvas: TCanvas);
    procedure pbxBtmBorderPaint(Sender: TObject; Canvas: TCanvas);
    procedure pbxTopBorderMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pbxTopBorderMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pbxTopBorderMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Single);
    procedure pbxTopBorderMouseLeave(Sender: TObject);
    procedure pbxBtmBorderMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pbxBtmBorderMouseLeave(Sender: TObject);
    procedure pbxBtmBorderMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Single);
    procedure pbxBtmBorderMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cmdTopClearClick(Sender: TObject);
    procedure cmdBtmClearClick(Sender: TObject);
    procedure cmdTopLoadClick(Sender: TObject);
    procedure cmdTopSaveClick(Sender: TObject);
    procedure cmdBtmLoadClick(Sender: TObject);
    procedure cmdBtmSaveClick(Sender: TObject);
    procedure pnlTopAMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlTopBMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlTopDMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlTopEMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlTopHMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlTopLMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlBtmAMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlBtmBMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlBtmDMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlBtmEMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlBtmHMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure pnlBtmLMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cboTopLBMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cboTopRBMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure optColorModeMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure optSizeModeMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cboBtmLBMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cboBtmRBMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cmdBtmClearMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cmdBtmLoadMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
    procedure cmdSaveASMClick(Sender: TObject);
    procedure cmdSaveDSKClick(Sender: TObject);
    procedure pnlPickColorMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Single);
  private
    TopBorder: BorderArray;
    BottomBorder: BorderArray;
    MouseButton: integer;
    function ColorIndexToRGB(iIndex: integer): cardinal;
    procedure PaintTopBorder;
    procedure PaintBtmBorder;
    procedure InsertColor(xChar, y, iCol: integer; var Border: BorderArray);
    procedure InsertNOP(xChar, y: integer; var Border: BorderArray);
    procedure DeleteNOP(xChar, y: integer; var Border: BorderArray);
    procedure ClearBorder(var Border: BorderArray);
    procedure CompressData(var Border: BorderArray);
    procedure ShowPickColorPanel(Sender: TPanel);
    function IsBorderClear(Border: BorderArray): boolean;
    procedure SaveDSK;
    procedure BuildASMLineInt(var sl: TStringList);
    { Private declarations }
  public
    { Public declarations }
  end;

var
  MainWnd: TMainWnd;

implementation

{$R *.fmx}

uses GrokUtils, Assemb, AsmTypes;

const
  PAL_NONE = 0;
  PAL_TOP_A = 1;
  PAL_TOP_B = 2;
  PAL_TOP_D = 3;
  PAL_TOP_E = 4;
  PAL_TOP_H = 5;
  PAL_TOP_L = 6;
  PAL_BTM_A = 7;
  PAL_BTM_B = 8;
  PAL_BTM_D = 9;
  PAL_BTM_E = 10;
  PAL_BTM_H = 11;
  PAL_BTM_L = 12;

var
  PickColorTarget: integer;

procedure SetEditControlColor(AItem: TListBoxItem; AColor: TAlphaColor);
var
  T: TFmxObject;
begin
  if AItem = nil then exit;

  T := AItem.FindStyleResource('background');
  if (T <> nil) and (T is TRectangle) then
     if TRectangle(T).Fill <> nil then
        TRectangle(T).Fill.Color := AColor;
  AItem.Repaint;
end;

procedure TMainWnd.lbiBtmColAPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlBtmA.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiBtmColBPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlBtmB.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiBtmColDPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlBtmD.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiBtmColEPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlBtmE.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiBtmColHPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlBtmH.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiBtmColLPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlBtmL.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiTopColAPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlTopA.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiTopColBPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlTopB.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiTopColDPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlTopD.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiTopColEPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlTopE.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;


procedure TMainWnd.lbiTopColHPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlTopH.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.lbiTopColLPaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  ColorRect: TRectF;
begin
  ColorRect := ARect;
  Canvas.Fill.Color :=  ColorIndexToRGB(pnlTopL.Tag);
  ColorRect.Left := ARect.Left + 52;
  ColorRect.Width := ARect.Left + 12;
  Canvas.FillRect(ColorRect,0,0,[],1,TCornerType.ctRound);
end;

procedure TMainWnd.optColorModeMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.optSizeModeMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.pbxBtmBorderMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
  if optColorMode.IsChecked then
  begin
    if (Button = TMouseButton.mbLeft) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboBtmLB.ItemIndex, BottomBorder);
      MouseButton := 1;
    end
    else if (Button = TMouseButton.mbRight) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboBtmRB.ItemIndex, BottomBorder);
      MouseButton := 2;
    end;
  end
  else begin
    // Size mode
    if (Button = TMouseButton.mbLeft) then
    begin
      InsertNOP(Round(x) div 16, Round(y) div 2, BottomBorder);
      MouseButton := 0;
    end
    else if (Button = TMouseButton.mbRight) then
    begin
      DeleteNOP(Round(x) div 16, Round(y) div 2, BottomBorder);
      MouseButton := 0;
    end;
  end;
  pbxBtmBorder.Repaint;
end;

procedure TMainWnd.pbxBtmBorderMouseLeave(Sender: TObject);
begin
  MouseButton := 0;
end;

procedure TMainWnd.pbxBtmBorderMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Single);
begin
  if (X >= 0) and (X < (Sender as TPaintBox).Width) and
     (Y >= 0) and (Y < (Sender as TPaintBox).Height) then
  begin
    if (MouseButton = 1) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboBtmLB.ItemIndex, BottomBorder);
      pbxBtmBorder.Repaint;
    end
    else if (MouseButton = 2) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboBtmRB.ItemIndex, BottomBorder);
      pbxBtmBorder.Repaint;
    end;
  end;
end;

procedure TMainWnd.pbxBtmBorderMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  MouseButton := 0;
end;

procedure TMainWnd.pbxBtmBorderPaint(Sender: TObject; Canvas: TCanvas);
begin
  PaintBtmBorder();
end;

procedure TMainWnd.InsertColor(xChar,y, iCol: integer; var Border: BorderArray);
var
  i,pos: integer;
begin
  pos := 0;

  for i := 0 to xChar do
  begin
    if Border[i,y] < 6 then
      inc(pos,2)
    else
      inc(pos,1);
    if pos > xChar then
    begin
      Border[i,y] := iCol;
      exit; // exit for loop
    end;
  end;
end;

procedure TMainWnd.InsertNOP(xChar,y: integer; var Border: BorderArray);
var
  i,j,pos: integer;
begin
  pos := 0;

  for i := 0 to xChar do
  begin
    if Border[i,y] < 6 then
      inc(pos,2)
    else
      inc(pos,1);
    if pos > xChar then
    begin
      for j := 46 downto i+1 do
        Border[j+1,y] := Border[j,y];
      Border[i+1,y] := 254;
      exit; // exit for loop
    end;
  end;
end;

procedure TMainWnd.DeleteNOP(xChar,y: integer; var Border: BorderArray);
var
  i,j,pos: integer;
begin
  pos := 0;

  for i := 0 to xChar do
  begin
    if Border[i,y] < 6 then
      inc(pos,2)
    else
      inc(pos,1);
    if (pos > xChar) and (Border[i+1,y] = 254) then
    begin
      for j := i+1 to 46 do
        Border[j,y] := Border[j+1,y];
      exit; // exit for loop
    end;
  end;
end;

procedure TMainWnd.FormMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.cmdBtmClearClick(Sender: TObject);
begin
  if MessageDlg('Clear bottom border image?',TMsgDlgType.mtConfirmation,mbOKCancel,0) = IDOK then
  begin
    ClearBorder(BottomBorder);
    pbxBtmBorder.Repaint;
  end;
end;

procedure TMainWnd.cmdBtmClearMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.cmdBtmLoadClick(Sender: TObject);
var
  F: TFileStream;
  cOut: array [0..5] of byte;
begin
  {$IFDEF MACOS}
  dlgLoad.FileName := '';
  {$ENDIF}

  pnlPickColor.Visible := false;
  if dlgLoad.Execute then
  begin
    if (STO_GetFileSize(dlgLoad.FileName) <> 390) and
       (STO_GetFileSize(dlgLoad.FileName) <> 1158) then
    begin
      MessageDlg('Invalid border file. Wrong file size.',TMsgDlgType.mtError,[TMsgDlgBtn.mbOK],0);
    end
    else
    begin
      ClearBorder(BottomBorder);

      F := TFileStream.Create(dlgLoad.FileName,fmOpenRead);
      F.Read(cOut,6);
      pnlBtmA.Tag := cOut[0];    pnlBtmA.Repaint;
      pnlBtmB.Tag := cOut[1];    pnlBtmB.Repaint;
      pnlBtmD.Tag := cOut[2];    pnlBtmD.Repaint;
      pnlBtmE.Tag := cOut[3];    pnlBtmE.Repaint;
      pnlBtmH.Tag := cOut[4];    pnlBtmH.Repaint;
      pnlBtmL.Tag := cOut[5];    pnlBtmL.Repaint;

      cboTopLB.Repaint;
      cboTopRB.Repaint;

      F.Read(BottomBorder,F.Size-6);
      FreeAndNil(F);

      CompressData(BottomBorder);
      pbxBtmBorder.Repaint;
    end;
  end;
end;

procedure TMainWnd.cmdBtmLoadMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.cmdBtmSaveClick(Sender: TObject);
var
  F: TFileStream;
  cOut: array [0..5] of byte;
begin
  pnlPickColor.Visible := false;
  CompressData(BottomBorder);
  pbxBtmBorder.Repaint;

  dlgSave.DefaultExt := 'bdr';
  dlgSave.Filter := 'Spectrum Border (*.bdr)|*.bdr|All Files (*.*)|*';
  dlgSave.Title := 'Save Border File';
  dlgSave.FileName := '';
  if dlgSave.Execute then
  begin
    F := TFileStream.Create(dlgSave.FileName,fmCreate or fmOpenWrite);
    cOut[0] := pnlBtmA.Tag;
    cOut[1] := pnlBtmB.Tag;
    cOut[2] := pnlBtmD.Tag;
    cOut[3] := pnlBtmE.Tag;;
    cOut[4] := pnlBtmH.Tag;;
    cOut[5] := pnlBtmL.Tag;
    F.Write(cOut,6);
    F.Write(BottomBorder,1152);
    FreeAndNil(F);
  end;
end;

procedure TMainWnd.cmdSaveASMClick(Sender: TObject);
var
  sl: TStringList;
begin
  CompressData(TopBorder);
  CompressData(BottomBorder);
  pbxTopBorder.Repaint;
  pbxBtmBorder.Repaint;

  sl := TStringList.Create();
  BuildASMLineInt(sl);

  dlgSave.DefaultExt := 'asm';
  dlgSave.Filter := 'Assembler Files (*.asm)|*.asm|All Files (*.*)|*.*';
  dlgSave.Title := 'Save Assembly Listing';
  dlgSave.FileName := '';
  if dlgSave.Execute then
    sl.SaveToFile(dlgSave.FileName);
  FreeAndNil(sl);
end;

procedure TMainWnd.cmdSaveDSKClick(Sender: TObject);
begin
  SaveDSK();
end;

procedure TMainWnd.cboBtmLBMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.cboBtmRBMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.cboTopLBMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.cboTopRBMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
end;

procedure TMainWnd.ClearBorder(var Border: BorderArray);
var
  x,y: integer;
begin
  for y := 0 to 24 do
  begin
    for x := 0 to 47 do
    begin
      Border[x,y] := 0; // Set to background
    end;
  end;
end;

procedure TMainWnd.pbxTopBorderMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  pnlPickColor.Visible := false;
  if optColorMode.IsChecked then
  begin
    if (Button = TMouseButton.mbLeft) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboTopLB.ItemIndex, TopBorder);
      MouseButton := 1;
    end
    else if (Button = TMouseButton.mbRight) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboTopRB.ItemIndex, TopBorder);
      MouseButton := 2;
    end;
  end
  else begin
    // Size mode
    if (Button = TMouseButton.mbLeft) then
    begin
      InsertNOP(Round(x) div 16, Round(y) div 2, TopBorder);
      MouseButton := 0;
    end
    else if (Button = TMouseButton.mbRight) then
    begin
      DeleteNOP(Round(x) div 16, Round(y) div 2, TopBorder);
      MouseButton := 0;
    end;
  end;
  pbxTopBorder.Repaint;
end;

procedure TMainWnd.pbxTopBorderMouseLeave(Sender: TObject);
begin
  MouseButton := 0;
end;

procedure TMainWnd.pbxTopBorderMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Single);
begin
  if (X >= 0) and (X < (Sender as TPaintBox).Width) and
     (Y >= 0) and (Y < (Sender as TPaintBox).Height) then
  begin
    if (MouseButton = 1) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboTopLB.ItemIndex, TopBorder);
      pbxTopBorder.Repaint;
    end
    else if (MouseButton = 2) then
    begin
      InsertColor(Round(x) div 16, Round(y) div 2, cboTopRB.ItemIndex, TopBorder);
      pbxTopBorder.Repaint;
    end;
  end;
end;

procedure TMainWnd.pbxTopBorderMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  MouseButton := 0;
end;

procedure TMainWnd.pbxTopBorderPaint(Sender: TObject; Canvas: TCanvas);
begin
  PaintTopBorder();
end;

procedure TMainWnd.cmdTopClearClick(Sender: TObject);
begin
  pnlPickColor.Visible := false;
  if MessageDlg('Clear top border image?',TMsgDlgType.mtConfirmation,mbOKCancel,0) = IDOK then
  begin
    ClearBorder(TopBorder);
    pbxTopBorder.Repaint;
  end;
end;

procedure TMainWnd.cmdTopLoadClick(Sender: TObject);
var
  F: TFileStream;
  cOut: array [0..5] of byte;
begin
  {$IFDEF MACOS}
  dlgLoad.FileName := '';
  {$ENDIF}

  pnlPickColor.Visible := false;
  if dlgLoad.Execute then
  begin
    if (STO_GetFileSize(dlgLoad.FileName) <> 390) and
       (STO_GetFileSize(dlgLoad.FileName) <> 1158) then
    begin
      MessageDlg('Invalid border file. Wrong file size.',TMsgDlgType.mtError,[TMsgDlgBtn.mbOK],0);
    end
    else
    begin
      ClearBorder(TopBorder);

      F := TFileStream.Create(dlgLoad.FileName,fmOpenRead);
      F.Read(cOut,6);
      pnlTopA.Tag := cOut[0];    pnlTopA.Repaint;
      pnlTopB.Tag := cOut[1];    pnlTopB.Repaint;
      pnlTopD.Tag := cOut[2];    pnlTopD.Repaint;
      pnlTopE.Tag := cOut[3];    pnlTopE.Repaint;
      pnlTopH.Tag := cOut[4];    pnlTopH.Repaint;
      pnlTopL.Tag := cOut[5];    pnlTopL.Repaint;

      cboTopLB.Repaint;
      cboTopRB.Repaint;

      F.Read(TopBorder,F.Size-6);
      FreeAndNil(F);

      CompressData(TopBorder);
      pbxTopBorder.Repaint;
    end;
  end;
end;

procedure TMainWnd.cmdTopSaveClick(Sender: TObject);
var
  F: TFileStream;
  cOut: array [0..5] of byte;
begin
  pnlPickColor.Visible := false;
  CompressData(TopBorder);
  pbxTopBorder.Repaint;

  dlgSave.DefaultExt := 'bdr';
  dlgSave.Filter := 'Spectrum Border (*.bdr)|*.bdr|All Files (*.*)|*';
  dlgSave.Title := 'Save Border File';
  dlgSave.FileName := '';
  if dlgSave.Execute then
  begin
    F := TFileStream.Create(dlgSave.FileName,fmCreate or fmOpenWrite);
    cOut[0] := pnlTopA.Tag;
    cOut[1] := pnlTopB.Tag;
    cOut[2] := pnlTopD.Tag;
    cOut[3] := pnlTopE.Tag;;
    cOut[4] := pnlTopH.Tag;;
    cOut[5] := pnlTopL.Tag;
    F.Write(cOut,6);
    F.Write(TopBorder,1152);
    FreeAndNil(F);
  end;
end;

function TMainWnd.ColorIndexToRGB(iIndex: integer): cardinal;
begin
  // RGB values for each of the SAM Coup's default
  // colours, nicked from SimCoupe
  case iIndex of
  0: Result := $FF000000;
  1: Result := $FF000092;
  2: Result := $FF920000;
  3: Result := $FF920092;
  4: Result := $FF009200;
  5: Result := $FF009292;
  6: Result := $FF929200;
  7: Result := $FF929292;
  32: Result := $FF010101;
  33: Result := $FF0000DB;
  34: Result := $FFDB0000;
  35: Result := $FFDB00DB;
  36: Result := $FF00DB00;
  37: Result := $FF00DBDB;
  38: Result := $FFDBDB00;
  39: Result := $FFDBDBDB;
  else
    Result := $FF000000;
  end;
end;

procedure TMainWnd.pnlPickColorMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  case PickColorTarget of
  PAL_TOP_A:  begin
                pnlTopA.Tag := (Sender as TPanel).Tag;
                cboTopLB.ItemIndex := 0;
              end;
  PAL_TOP_B:  begin
                pnlTopB.Tag := (Sender as TPanel).Tag;
                cboTopLB.ItemIndex := 1;
              end;
  PAL_TOP_D:  begin
                pnlTopD.Tag := (Sender as TPanel).Tag;
                cboTopLB.ItemIndex := 2;
              end;
  PAL_TOP_E:  begin
                pnlTopE.Tag := (Sender as TPanel).Tag;
                cboTopLB.ItemIndex := 3;
              end;
  PAL_TOP_H:  begin
                pnlTopH.Tag := (Sender as TPanel).Tag;
                cboTopLB.ItemIndex := 4;
              end;
  PAL_TOP_L:  begin
                pnlTopL.Tag := (Sender as TPanel).Tag;
                cboTopLB.ItemIndex := 5;
              end;
  PAL_BTM_A:  begin
                pnlBtmA.Tag := (Sender as TPanel).Tag;
                cboBtmLB.ItemIndex := 0;
              end;
  PAL_BTM_B:  begin
                pnlBtmB.Tag := (Sender as TPanel).Tag;
                cboBtmLB.ItemIndex := 1;
              end;
  PAL_BTM_D:  begin
                pnlBtmD.Tag := (Sender as TPanel).Tag;
                cboBtmLB.ItemIndex := 2;
              end;
  PAL_BTM_E:  begin
                pnlBtmE.Tag := (Sender as TPanel).Tag;
                cboBtmLB.ItemIndex := 3;
              end;
  PAL_BTM_H:  begin
                pnlBtmH.Tag := (Sender as TPanel).Tag;
                cboBtmLB.ItemIndex := 4;
              end;
  PAL_BTM_L:  begin
                pnlBtmL.Tag := (Sender as TPanel).Tag;
                cboBtmLB.ItemIndex := 5;
              end;
  end;

  pnlTopA.Repaint;
  pnlTopB.Repaint;
  pnlTopD.Repaint;
  pnlTopE.Repaint;
  pnlTopH.Repaint;
  pnlTopL.Repaint;
  pnlBtmA.Repaint;
  pnlBtmB.Repaint;
  pnlBtmD.Repaint;
  pnlBtmE.Repaint;
  pnlBtmH.Repaint;
  pnlBtmL.Repaint;
  cboTopLB.Repaint;
  cboTopRB.Repaint;
  cboBtmLB.Repaint;
  cboBtmRB.Repaint;
  pbxTopBorder.Repaint;
  pbxBtmBorder.Repaint;

  pnlPickColor.Visible := false;
  PickColorTarget := PAL_NONE;
end;

procedure TMainWnd.pnlBtmAMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_BTM_A;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlBtmBMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_BTM_B;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlBtmDMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_BTM_D;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlBtmEMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_BTM_E;
  ShowPickColorPanel(Sender as TPanel);

end;

procedure TMainWnd.pnlBtmHMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_BTM_H;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlBtmLMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_BTM_L;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlColorPalettePaint(Sender: TObject; Canvas: TCanvas;
  const ARect: TRectF);
var
  CRect: TRectF;
begin
  CRect.Left := ARect.Left + 1;
  CRect.Width := ARect.Width - 2;
  CRect.Top := ARect.Top + 1;
  CRect.Height := ARect.Height - 2;

  Canvas.Fill.Kind := TBrushKind.bkSolid;
  Canvas.Fill.Color := ColorIndexToRGB((Sender as TPanel).Tag);
  Canvas.FillRect(CRect,0,0,[],1);
end;

procedure TMainWnd.ShowPickColorPanel(Sender: TPanel);
begin
  pnlPickColor.Parent := Sender.Parent;
  pnlPickColor.Position.X := Sender.Position.X;
  pnlPickColor.Position.Y := Sender.Position.Y + Sender.Height;
  pnlPickColor.Visible := true;
end;

procedure TMainWnd.pnlTopAMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_TOP_A;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlTopBMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_TOP_B;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlTopDMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_TOP_D;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlTopEMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_TOP_E;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlTopHMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_TOP_H;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.pnlTopLMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Single);
begin
  PickColorTarget := PAL_TOP_L;
  ShowPickColorPanel(Sender as TPanel);
end;

procedure TMainWnd.PaintTopBorder();
var
  x,y: integer;
  iCol: Cardinal;
  k: Integer;
  iPixelX: integer;
  bmp: TBitmap;
  RSrc, RDst: TRectF;
begin
  bmp := TBitmap.Create(320,24);

  iCol := pnlTopA.Tag; // Default to background colour
  for y := 0 to 23 do
  begin
    iPixelX := 0;
    for x := 0 to 47 do
    begin
      case TopBorder[x,y] of
      0: // Background
        iCol := pnlTopA.Tag;
      1: // Col1
        iCol := pnlTopB.Tag;
      2: // Col2
        iCol := pnlTopD.Tag;
      3: // Col3
        iCol := pnlTopE.Tag;
      4: // Col4
        iCol := pnlTopH.Tag;
      5: // Col5
        iCol := pnlTopL.Tag;
      end;

      if TopBorder[x,y] < 6 then
      begin
        iCol := ColorIndexToRGB(iCol);
        {$IFDEF MACOS}
          // The Pixels member of bitmap has Red and Blue swapped in OSX
          iCol := (iCol and $FF00FF00) or
                  ((iCol and $FF0000) shr 16) or
                  ((iCol and $FF) shl 16);
        {$ENDIF}
        for k := iPixelX to iPixelX + 15 do
        begin
          bmp.Pixels[k,y] := iCol;
        end;
        inc(iPixelX,16);
      end
      else if TopBorder[x,y] = 254 then
      begin
        // Nop
        // Use iCol value from previous loop iteration
        for k := iPixelX to iPixelX + 7 do
        begin
         bmp.Pixels[k,y] := iCol;
        end;
        inc(iPixelX,8);
      end;
      if iPixelX > 320 then break;
    end;
  end;

  RSrc := RectF(0,0,bmp.Width,bmp.Height);
  RDst := RectF(0,0,640,48);
  pbxTopBorder.Canvas.DrawBitmap(bmp,RSrc,RDst,1,true);   ///  StretchDraw(R,bmp);
  FreeAndNil(bmp);
end;

procedure TMainWnd.PaintBtmBorder();
var
  x,y: integer;
  iCol: Cardinal;
  k: Integer;
  iPixelX: integer;
  bmp: TBitmap;
  RSrc, RDst: TRectF;
begin
  bmp := TBitmap.Create(320,24);

  iCol := pnlBtmA.Tag; // Default to background colour
  for y := 0 to 23 do
  begin
    iPixelX := 0;
    for x := 0 to 47 do
    begin
      case BottomBorder[x,y] of
      0: // Background
        iCol := pnlBtmA.Tag;
      1: // Col1
        iCol := pnlBtmB.Tag;
      2: // Col2
        iCol := pnlBtmD.Tag;
      3: // Col3
        iCol := pnlBtmE.Tag;
      4: // Col4
        iCol := pnlBtmH.Tag;
      5: // Col5
        iCol := pnlBtmL.Tag;
      end;

      if BottomBorder[x,y] < 6 then
      begin
        iCol := ColorIndexToRGB(iCol);
        {$IFDEF MACOS}
          // The Pixels member of bitmap has Red and Blue swapped in OSX
          iCol := (iCol and $FF00FF00) or
                  ((iCol and $FF0000) shr 16) or
                  ((iCol and $FF) shl 16);
        {$ENDIF}
        for k := iPixelX to iPixelX + 15 do
        begin
          bmp.Pixels[k,y] := iCol;
        end;
        inc(iPixelX,16);
      end
      else if BottomBorder[x,y] = 254 then
      begin
        // Nop
        // Use iCol value from previous loop iteration
        for k := iPixelX to iPixelX + 7 do
        begin
         bmp.Pixels[k,y] := iCol;
        end;
        inc(iPixelX,8);
      end;
      if iPixelX > 320 then break;
    end;
  end;

  RSrc := RectF(0,0,bmp.Width,bmp.Height);
  RDst := RectF(0,0,640,48);
  pbxBtmBorder.Canvas.DrawBitmap(bmp,RSrc,RDst,1,true);   ///  StretchDraw(R,bmp);
  FreeAndNil(bmp);
end;

procedure TMainWnd.CompressData(var Border: BorderArray);
var
  x,y: integer;
  iCol: integer;
  i: Integer;
begin
  for y := 0 to 23 do
  begin
    x := 0;
    iCol := 0;
    while x < 47 do
    begin
      if Border[x,y] < 6 then
        iCol := Border[x,y];
      if (Border[x,y] = 254) and
         (Border[x+1,y] = 254) then
      begin
        Border[x,y] := iCol;
        for i := x+1 to 46 do
          Border[i,y] := Border[i+1,y];
        Border[47,y] := 0;
      end;
      inc(x);
    end;
  end;
end;

procedure TMainWnd.BuildASMLineInt(var sl: TStringList);
var
  iOrg: integer;
  x,y: integer;
  iTCount: integer;
  iTTarget: integer;
begin
  sl.Add('; Created by "BorderTron3000 - Sam Coupe Edition"');
  sl.Add('; By Chris Cowley. <chrrris@gmail.com>');
  sl.Add('; Please use this code freely.');
  sl.Add('');
  sl.Add('; Updated for SAM Coupe timings with much assistance from Simon Owen');
  sl.Add('');
  iOrg := StrToIntDef(txtOrg.Text,-1);
  if (iOrg < 0) or (iOrg > 65023) then
    iOrg := 40000;

  sl.Add('                  org   ' + IntToStr(iOrg));
  sl.Add('Start:            call  SetupIntHandler');
  if not IsBorderClear(BottomBorder) then
  begin
    // Set up a line interrupt if we have an image in the bottom border
    sl.Add('                  ld    a,191');
    sl.Add('                  out   (249),a  ; setup line interrupt');
  end;
  sl.Add('                  ei');
  sl.Add('');
  sl.Add('BorderLoop:');
  sl.Add('                  halt           ; borders are drawn in the interrupt');
  sl.Add('');
  sl.Add('                  ; If you want the border effects to be active during');
  sl.Add('                  ; some other processing (such as during a game''s main loop)');
  sl.Add('                  ; This would be an ideal location to place the CALL.');
  sl.Add('                  ; You don''t need to preserve any registers.');
  sl.Add('');
  if chkExitKeypress.IsChecked then
  begin
    sl.Add('                  xor   a');
    sl.Add('                  in    a,($FE)');
    sl.Add('                  and   $1F');
    sl.Add('                  cp    $1F');
    sl.Add('                  jr    z,BorderLoop');
    sl.Add('');
    sl.Add('ReturnToBASIC:    di');
    sl.Add('                  xor   a');
    sl.Add('                  ld    a,i');
    sl.Add('                  im    1');
    sl.Add('                  cpl            ;');
    sl.Add('                  out   (249),a  ; disable line interrupt');
    sl.Add('                  ei');
    sl.Add('                  ret');
  end
  else
    sl.Add('                  jr    BorderLoop');

  sl.Add('');
  sl.Add('DrawTopBorder:');
  sl.Add('                  ; Setup registers for our top border image');
  sl.Add('                  ; C holds the output port (&FE) so is out of bounds');
  sl.Add('                  ld    a,' + IntToStr(pnlTopA.Tag));
  sl.Add('                  ld    hl,$' + Format('%.2x%.2x',[pnlTopH.Tag,pnlTopL.Tag]));
  sl.Add('                  ld    de,$' + Format('%.2x%.2x',[pnlTopD.Tag,pnlTopE.Tag]));
  sl.Add('                  ld    bc,$' + Format('%.2x%.2x',[pnlTopB.Tag,$FE]));
  sl.Add('');
  sl.Add('                  out   (c),a    ; background colour');
  sl.Add('                  ld    bc,$1705');
  sl.Add('WaitLoop1:        djnz  WaitLoop1');
  sl.Add('                  dec   c');
  sl.Add('                  jr    nz,WaitLoop1');
  sl.Add('');
  sl.Add('                  ld    bc,$' + Format('%.2x%.2x',[pnlTopB.Tag,$FE]));

  for y := 0 to 23 do
  begin
    sl.Add('                  ; Row ' + IntToStr(y));
    iTCount := 0;
    for x := 0 to 47 do
    begin
      case TopBorder[x,y] of
      0: // Background
        begin
          sl.Add('                  out   (c),a'); // 23
          inc(iTCount,16);
        end;
      1: // Col1
        begin
          sl.Add('                  out   (c),b');
          inc(iTCount,16);
        end;
      2: // Col2
        begin
          sl.Add('                  out   (c),d');
          inc(iTCount,16);
        end;
      3: // Col3
        begin
          sl.Add('                  out   (c),e');
          inc(iTCount,16);
        end;
      4: // Col4
        begin
          sl.Add('                  out   (c),h');
          inc(iTCount,16);
        end;
      5: // Col5
        begin
          sl.Add('                  out   (c),l');
          inc(iTCount,16);
        end;
      254: // NOP
        begin
          sl.Add('                  nop');
          inc(iTCount,8);
        end;
      end;
      if iTCount >= 320 then break;
    end;

    // The code below places at least one OUT (C),A at the end of every
    // scanline, this serves the useful purpose of ensuring the colour from the
    // bottom-right of the image doesn't bleed down into the left&right border
    // areas, but it also makes the image look neat and tidy in SimCoupe's
    // View->Complete Border mode. We have to do something to use up the
    // remaining cycles to the end of each scanline, so this is as good as
    // anything :)
    iTTarget := 384;

    while (iTCount < iTTarget) do
    begin
      if iTCount <= (iTTarget-16) then
      begin
        sl.Add('                  out   (c),a  ; HIDDEN - Background');
        inc(iTCount,16);
      end
      else begin
        sl.Add('                  nop');
        inc(iTCount,8);
      end;
    end;

    sl.Add('');
  end;

  sl.Add('                  ei');
  sl.Add('                  ret');
  sl.Add('');
  sl.Add('ISRMain:');
  sl.Add('                  in    a,(249)');
  sl.Add('                  rrc   a         ; test for line int');
  sl.Add('                  jr    nc,DrawBottomBorder');
  sl.Add('                  jp    DrawTopBorder');
  sl.Add('');
  sl.Add('DrawBottomBorder:');
  if not IsBorderClear(BottomBorder) then
  begin
    sl.Add('                  ; Setup registers for our bottom border image');
    sl.Add('                  ; C holds the output port ($FE) so is out of bounds');
    sl.Add('                  ld    a,' + IntToStr(pnlBtmA.Tag));
    sl.Add('                  ld    bc,$' + Format('%.2x%.2x',[pnlBtmB.Tag,$FE]));
    sl.Add('                  ld    hl,$' + Format('%.2x%.2x',[pnlBtmH.Tag,pnlBtmL.Tag]));
    sl.Add('                  ld    de,$' + Format('%.2x%.2x',[pnlBtmD.Tag,pnlBtmE.Tag]));
    sl.Add('');
    sl.Add('                  ld    b,$14');
    sl.Add('WaitLoop2:        djnz  WaitLoop2');
    sl.Add('');
    sl.Add('                  ld    b,$' + Format('%.2x',[pnlBtmB.Tag]));
    sl.Add('');

    for y := 0 to 23 do
    begin
      sl.Add('                  ; Bottom Row ' + IntToStr(y));
      iTCount := 0;
      for x := 0 to 47 do
      begin
        case BottomBorder[x,y] of
        0: // Background
          begin
            sl.Add('                  out   (c),a'); // 23
            inc(iTCount,16);
          end;
        1: // Col1
          begin
            sl.Add('                  out   (c),b');
            inc(iTCount,16);
          end;
        2: // Col2
          begin
            sl.Add('                  out   (c),d');
            inc(iTCount,16);
          end;
        3: // Col3
          begin
            sl.Add('                  out   (c),e');
            inc(iTCount,16);
          end;
        4: // Col4
          begin
            sl.Add('                  out   (c),h');
            inc(iTCount,16);
          end;
        5: // Col5
          begin
            sl.Add('                  out   (c),l');
            inc(iTCount,16);
          end;
        254: // NOP
          begin
            sl.Add('                  nop');
            inc(iTCount,8);
          end;
        end;
        if iTCount >= 320 then break;
      end;
      iTTarget := 384;

      // See the comment above, in the DrawTopBorder: routine,about putting
      // the OUT (C),A's at the end of each scanline
      while (iTCount < iTTarget) do
      begin
        if iTCount <= (iTTarget-16) then
        begin
          sl.Add('                  out   (c),a  ; HIDDEN - Background');
          inc(iTCount,16);
        end
        else begin
          sl.Add('                  nop');
          inc(iTCount,8);
        end;
      end;
      sl.Add('');
    end;
    sl.Add('                  ei');
    sl.Add('                  ret');
  end
  else
  begin
   sl.Add('                  ; The JR to DrawBottomBorder is never made');
   sl.Add('                  ; as there is no bottom border image. It has');
   sl.Add('                  ; been left in purely to maintain timing.');
  end;

  sl.Add('');
  sl.Add('SetupIntHandler:  di');
  sl.Add('                  ; Put JR at $FFFF = JR to $FFF4');
  sl.Add('                  ld    a,$18');
  sl.Add('                  ld    ($FFFF),a');
  sl.Add('');
  sl.Add('                  ; Put a JP ISRMain at $FFF4');
  sl.Add('                  ld    hl,$FFF4');
  sl.Add('                  ld    (hl),$C3   ; JP nn');
  sl.Add('                  inc   hl');
  sl.Add('                  ld    (hl),ISRMain ^ 256   ; ^ is modulo (some assemblers use a different syntax though)');
  sl.Add('                  inc   hl');
  sl.Add('                  ld    (hl),ISRMain / 256');
  sl.Add('');
  sl.Add('                  ; Put interrupt vector table at $FE00');
  sl.Add('                  ld    hl,$FE00');
  sl.Add('                  ld    (hl),$FF');
  sl.Add('                  ld    de,$FE01');
  sl.Add('                  ld    bc,$0100');
  sl.Add('                  ldir');
  sl.Add('                  ld    a,$FE');
  sl.Add('                  ld    i,a');
  sl.Add('                  im    2');
  sl.Add('                  ret');
end;

function TMainWnd.IsBorderClear(Border: BorderArray): boolean;
var
  x: integer;
  y: Integer;
begin
  Result := true;

  for y := 0 to 23 do
  begin
    for x := 0 to 47 do
    begin
      if Border[x,y] <> 0 then
      begin
        Result := false;
        exit;
      end;
    end;
  end;
end;

procedure TMainWnd.SaveDSK();
var
  sl: TStringList;
  sFileName: string;
  iLen, iOrg, iSector, iWritten: integer;
  F: TFileStream;
  cOut,cCode: array [0..65538] of byte;
  i: Integer;
begin
  dlgSave.DefaultExt := 'dsk';
  dlgSave.Filter := 'SAM DSK Files (*.dsk)|*.dsk|All Files (*.*)|*.*';
  dlgSave.Title := 'Save DSK File';
  dlgSave.FileName := '';
  if dlgSave.Execute then
  begin
    sl := TStringList.Create();

    BuildASMLineInt(sl);
    ReadASMTableFile(ExtractFilePath(ParamStr(0)) + '\z80.tab');
    sFileName := dlgSave.FileName;

    // Create an empty file
    F := TFileStream.Create(sFileName,fmCreate or fmShareDenyWrite);
    //FreeAndNil(F);

    //F := TFileStream.Create(sFileName,fmOpenWrite or fmShareDenyWrite);
    //F.Seek(0,soFromEnd);

    // DSK file with autoboot
    iLen := AssembleStringListMem(sl,cCode,ASCII);

    // TODO: Add SAMDOS2 image to the disk structure :)

    // DSK Autoboot header
    cOut[0] := $13; // length
    cOut[1] := $41;
    cOut[2] := $55;
    cOut[3] := $54;
    cOut[4] := $4F; // "AUTO"
    COut[5] := Ord('B');
    COut[6] := Ord('d');
    COut[7] := Ord('r');
    COut[8] := Ord('T');
    COut[9] := Ord('r');
    COut[10] := Ord('n');
    COut[11] := $00;
    if iLen <= 501 then
      cOut[12] := $01 // 1 sector is enough for 501 bytes of code
    else
      cOut[12] := 2 + (iLen - 501) div 510;

    COut[12] := $05;  // Number of sectors
    COut[13] := $04;
    COut[14] := $01;
    COut[15] := $1F;
    F.Write(cOut[0],16);

    for i := 0 to 65538 do
      cOut[i] := 0;

    F.Write(cOut[0],220);

    iOrg := StrToIntDef(txtOrg.Text,-1);
    if (iOrg < 0) or (iOrg > 65320) then iOrg := 40000;

    cOut[0] := $01;
    cOut[1] := iOrg mod 256;
    cOut[2] := iOrg div 256;
    cOut[3] := 0;
    cOut[4] := iLen mod 256;
    cOut[5] := iLen div 256;
    cOut[6] := $02;  // Offset 000000F2
    cOut[7] := iOrg mod 256;
    cOut[8] := iOrg div 256;

    F.Write(cOut[0],40724);  // This takes us up to A000

    cOut[0] := $13;
    cOut[1] := iLen mod 256;
    cOut[2] := iLen div 256;
    cOut[3] := iOrg mod 256;
    cOut[4] := iOrg div 256;
    cOut[5] := $00;
    cOut[6] := $00;
    cOut[7] := $00;
    cOut[8] := $01;

    F.Write(cOut[0],9);  // This takes us up to A009
    iWritten := $A009;
    // DSK file with autoboot

    // Write first sector
    iSector := 1;
    if iLen <= 501 then
    begin
      F.Write(cCode[0],iLen);
      i := iLen;
    end
    else begin
      F.Write(cCode[0],501);
      i := 501;
    end;
    inc(iWritten,i);

    // Write subsequent sectors
    while i < iLen do
    begin
      inc(iSector);
      cOut[0] := 04;
      cOut[1] := iSector;
      F.Write(cOut[0],2);   // Sector shit
      F.Write(cCode[i],510); // Next 510 of 2160 bytes
      inc(iWritten,512);
      inc(i,510);
    end;

    for i := 0 to 65538 do
      cOut[i] := 0;

    while iWritten < 819200 do
    begin
      if (819200 - iWritten) >= 65536 then
      begin
        F.Write(cOut[0],65536);
        inc(iWritten,65536);
      end
      else begin
        F.Write(cOut[0],819200-iWritten);
        inc(iWritten,819200-iWritten);
      end;
    end;

    FreeAndNil(F);
    FreeAndNil(sl);
  end;
end;

end.

